/*
	File:		KodakDiconixDriver.cp

	Contains:	Newton printer driver for the Kodak Diconix family.
				This is an implementation of the TDotPrinterDriver protocol.

	Copyright:	 1992-1994 by Apple Computer, Inc., all rights reserved.

*/

#ifndef __KODAKDRIVER_H
	#include "KodakDiconixDriver.impl.h"				// **Master File Name**  (.impl.h)
#endif

#ifndef __string_h
	#include "string.h"
#endif

#ifndef __stdio_h
	#include "stdio.h"
#endif

#ifndef	__COMMMANAGERINTERFACE_H
	#include "CommManagerInterface.h"
#endif

#ifndef	__COMMSERVICES_H
	#include "CommServices.h"
#endif

#ifndef __ENDPOINT_H
	#include "Endpoint.h"
#endif

#ifndef	__SERIALOPTIONS_H
	#include "SerialOptions.h"
#endif




PROTOCOL_IMPL_SOURCE_MACRO(DRIVERCLASSNAME)



/*----- Constants and Enums -----*/

#define StartPageCommands		"\0331011\0336001\0336020\0336060"
													// Unidirectional, Graphics mode
													// column by column (no repeat)
#define Extra701OpenPage		"\0336040075"		// Move Paper down 1/4"
#define FinishPageCommands		"\014"				// form feed

#define GraphicsRowTransfer		"\033601"			// Start Graphics
#define BlankLine				"\0336010000"		// Nothing on line


const TTimeout kSndTimeout = 5 * kSeconds;


/*----- Globals -----*/

//  IMPORTANT:  LOADED PROTOCOLS MAY NOT HAVE GLOBALS
//	If you have driver-global data, it belongs in your object.



/*----- Driver Functions -----*/


/*-----------------------------------------------------------------------------
	NewtonErr Open(void)

	Initialize and Open the driver.
	
	Remember, if this call fails, no other calls will follow.  So you must
	clean up anything you do here (especially allocations).
	
	A good general structure for Open() might be:
	
		1.  NIL out allocations & status fields
		2.  Set up other variables
		3.  Try to open the connection
		4.  If it fails, clean up allocations
   -----------------------------------------------------------------------------*/
NewtonErr DRIVERCLASSNAME::Open(void)
{
	fAsyncIntf = nil;
	fCurrentError = noErr;

	PRINT(("DRIVERCLASSNAME::Open"));
	
	{		// check user-specified psuedo-driver
		RefVar	routingSlip = fConnect->fConnectInfo;
		RefVar	prRef = GetFrameSlot(routingSlip, SYM(printer));
		RefVar	modelRef = GetFrameSlot(prRef, SYM(prmodel));
		fHWType = (KodakVersion) RINT(modelRef);
	}

	if (!InitConnection())
		fCurrentError = kPR_ERR_NewtonError;

	if (fCurrentError)
		Cleanup();
		
	return fCurrentError;
}


/*-----------------------------------------------------------------------------
	void Delete(void)

	Historical entry point.  Probably shouldn't be used.
   -----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::Delete(void)
{
}




/*-----------------------------------------------------------------------------
	void Cleanup(void)

	Generally, this would be called by:
		1.  Open failure
		2.  Close
   -----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::Cleanup(void)
{
	if (fAsyncIntf)
	{
		fAsyncIntf->Delete();
		fAsyncIntf = nil;
	}
}


/*-----------------------------------------------------------------------------
	NewtonErr Close(void)

	Close the driver.  This call should follow this format:

		send final commands
		close connection
		cleanup allocations
   -----------------------------------------------------------------------------*/
NewtonErr DRIVERCLASSNAME::Close()
{
	PRINT(("DRIVERCLASSNAME::Close"));
	ReleaseConnection();
	Cleanup();
	return fCurrentError;
}




/*-----------------------------------------------------------------------------
	NewtonErr OpenPage(void)

	Send any commands required to begin a new page.
   -----------------------------------------------------------------------------*/
NewtonErr DRIVERCLASSNAME::OpenPage()
{
	PRINT(("DRIVERCLASSNAME::OpenPage"));

	SendString(StartPageCommands);
	if (fHWType == k701)
		SendString(Extra701OpenPage);
	return fCurrentError;
}





/*-----------------------------------------------------------------------------
	NewtonErr ClosePage(void)

	Send any commands required to finish a page.
   -----------------------------------------------------------------------------*/
NewtonErr DRIVERCLASSNAME::ClosePage()
{
	PRINT(("DRIVERCLASSNAME::ClosePage"));
	SendString(FinishPageCommands);
	return fCurrentError;
}




/*-----------------------------------------------------------------------------
	NewtonErr ImageBand(PixelMap* theBand, const Rect* minBounds)
	
	Printing the bitmap on a Kodak Diconix requires that you rotate the bits into
	a horizontal position before sending them.  The following is bit rotation code.

	To speed up the transmission process the output is buffered until a full line
	of graphics is processed and then this line is sent to the printer.
   -----------------------------------------------------------------------------*/
NewtonErr DRIVERCLASSNAME::ImageBand(PixelMap* theBand, const Rect* minBounds)
{
	Long			numpasses, pass, totalpasses;
	Long			skiptop, skipbottom;
	Long			numsegs, segs, n, m, p, numbytes;
	Long			count = 0;
	Long			pins, pinBytes;
	Ptr				srcBptr, segptr;
	ULong			pindata[50];
	UChar			dstData[7];
	UChar			size[5];
	UChar			buffer[224];

	PRINT(("ImageBand"));

	if (fCurrentError)
		return fCurrentError;

	if (fHWType == k180si)
		pins = 12;
	else
		pins = 50;

	srcBptr = theBand->baseAddr;
	numbytes = theBand->rowBytes;
	numsegs = (((minBounds->right - theBand->bounds.left)>>3) + 3) >> 2;
	skiptop = (minBounds->top - theBand->bounds.top)/pins;
	skipbottom = (theBand->bounds.bottom - minBounds->bottom)/pins;
	totalpasses = (theBand->bounds.bottom - theBand->bounds.top)/pins;
	numpasses = totalpasses - skiptop - skipbottom;
	pinBytes = (pins + 7) >> 3;
	sprintf((Ptr) size,"%04d",numsegs<<5);  /* should test to see what the number is */

	if (! (minBounds->bottom - minBounds->top)) 
	{
		/* minBounds empty, set the appropriate variables.*/
		skiptop = totalpasses;
		numpasses = skipbottom = 0;
	}

	PRINT(("bytes: %d, passes: %d, segments: %d, pinBytes: %d, size: %s",
								numbytes, numpasses, numsegs, pinBytes, size));

	for (pass=0; pass<skiptop; ++pass)
	{
		SendString(BlankLine);
		srcBptr += numbytes*pins;
	}

	for (pass=0; pass<numpasses; ++pass)
	{
		SendString(GraphicsRowTransfer);
		SendString((Ptr) size);
		for (segs=0; segs<numsegs; ++segs)
		{
			segptr = srcBptr + (segs << 2);
			for (n=0; n<pins; ++n)
			{
				pindata[n] = *(ULong *)segptr;
				segptr += numbytes;
			}
			for (n=0; n<32; ++n)
			{
				for (m=0; m<pinBytes; ++m)
				{
					dstData[m] = 0;
					for (p=m*8; p<(m+1)*8; ++p) 
					{
						if (p < pins) {
							dstData[m] = (UChar) ((dstData[m] << 1) | (pindata[p] >> 31));
							pindata[p] <<= 1;
						} 
						else 
						{
							dstData[m] <<= 1;
						}
					}
					buffer[count] = dstData[m];
					count += 1;
				}
			}
			SendData((Ptr) buffer, count);
			count = 0;
		}
		srcBptr += numbytes*pins;
	}

	for (pass=0; pass<skipbottom; ++pass)
	{
		SendString(BlankLine);
		srcBptr += numbytes*pins;
	}

	PRINT(("Band complete."));
	return fCurrentError;
}



/*-----------------------------------------------------------------------------
	void	CancelJob(Boolean asyncCancel)

	Called by the print loop if we are going to early-abort a print job.
	The print loop will still call the appropriate close() calls.  This
	just warns the driver that something fishy is going on.
	
	When async is true, ONLY those drivers which do tricky asynchronous
	operations might abort them (recover control).

	The majority of drivers will only care about the synchronous cancels,
	when it's OK to do (for example) a page eject here.
   -----------------------------------------------------------------------------*/
void	DRIVERCLASSNAME::CancelJob(Boolean asyncCancel)
{
	if (!asyncCancel)
		fCurrentError = kPR_ERR_UserCancel;
}



/*-----------------------------------------------------------------------------
	PrProblemResolution	IsProblemResolved(void);

	Report on the possible resolution of the current problem or error.
	
	Since this printer has no auto-resolution, always return kPrProblemNotFixed.
   -----------------------------------------------------------------------------*/
PrProblemResolution	DRIVERCLASSNAME::IsProblemResolved(void)
{
	return kPrProblemNotFixed;
}



/*-----------------------------------------------------------------------------
	void GetPageInfo(PrPageInfo* result)

	Kodak Diconix 180si
		192 x 96 dpi
		8" across, 10.375" tall (1363 pixels horizontally)

	Kodak Diconix 701
		300 x 300 dpi
		8" across, 10.375" tall (2400 pixels horizontally)

	Caution:  The first item is an FPoint - x,y.  The second is a point - v, h.
   -----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::GetPageInfo(PrPageInfo* result)
{
	PrPageInfo	dummy;

	Boolean	a4Paper = EQ(fConnect->fPaperSize, SYM(a4));

	if (fHWType == k180si)
	{
		dummy.printerDPI.x = Long2Fixed(192);
		dummy.printerDPI.y = Long2Fixed(96);
		dummy.printerPageSize.v = (Short) 10.375*96;
		dummy.printerPageSize.h = (Short) (1363);
	} 
	else 
	{
		dummy.printerDPI.x = Long2Fixed(300);
		dummy.printerDPI.y = Long2Fixed(300);
		dummy.printerPageSize.v = (Short) 10.375*300;
		dummy.printerPageSize.h = (Short) (2400);
	}

	*result = dummy;
}


/*-----------------------------------------------------------------------------
	void GetBandPrefs(DotPrinterPrefs* result)
	
	See the driver documentation - band size is a very important (and somewhat
	tricky) value and you really want to get it right.

	For the Diconix180si, the minimum band size is 12 (24 for 192 vertical dpi).
	
	For the Diconix701, the minimum band size is 50 and the optimum is also 50.
	This is quite a bit bigger than the average band.  It's big because the head
	is 50 pins high.

	The driver is synchronous - don't need async bands.
	Min bounds are useful.
   -----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::GetBandPrefs(DotPrinterPrefs* result)
{
	DotPrinterPrefs	dummy;

	if (fHWType == k180si)
	{
		dummy.minBand = 12;						// 12 is OK for 96 dpi
		dummy.optimumBand = 96;
		dummy.asyncBanding = false;
		dummy.wantMinBounds = true;
	} 
	else 
	{
		dummy.minBand = 50;						// 50 because of 50 pin head
		dummy.optimumBand = 50;
		dummy.asyncBanding = false;
		dummy.wantMinBounds = true;
	}

	*result = dummy;
}


/*-----------------------------------------------------------------------------
	void SendData(Ptr data, long length)
	
	Send a block of characters to the printer.
	-----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::SendData(Ptr data, long length)
{
	Size count = length;
	NewtonErr err;

	err = fAsyncIntf->Snd((UByte*)data, count, 0, kSndTimeout);
	if (err == kError_Call_Aborted)
		fCurrentError = kPR_ERR_UserCancel;
	else if (err == kError_Message_Timed_Out)	// this means we blocked,
		fCurrentError = kPR_ERR_PrinterError;	// probably due to hshk problem?
}


/*-----------------------------------------------------------------------------
	void SendString(char* str)
	
	Send a C-string to the printer.
	-----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::SendString(char* str)
{
	SendData((Ptr) str, strlen(str));
}


/*-----------------------------------------------------------------------------
	void ReleaseConnection()
	
	Close the serial connection.
	-----------------------------------------------------------------------------*/
void DRIVERCLASSNAME::ReleaseConnection()
{
	if (fAsyncIntf)
		fCurrentError = fAsyncIntf->EasyClose();
}


/*-----------------------------------------------------------------------------
	Boolean InitConnection();

	There may be problems with the flow control for this driver which is
	showing up now due to the addition of white space stripping and increased
	printing speed.

	Returns true if the async connection is successfully created.
	-----------------------------------------------------------------------------*/
Boolean DRIVERCLASSNAME::InitConnection()
{
	Boolean	initFailed = false;

	TOptionArray options;
	NewtonErr result = options.Init();

	if (result == noErr)				// choose raw async as the transport
	{
		TOption service;
		service.SetAsService(kCMSAsyncSerial);
		result = options.AppendOption(&service);
	}

	if (result == noErr)				// set speed
	{
		TCMOSerialIOParms ioOpt;
		if (fHWType == k180si)
			ioOpt.fSpeed = k9600bps;
		else
			ioOpt.fSpeed = k57600bps;

		result = options.AppendOption(&ioOpt);
	}

	if (result == noErr)				// set output flow control
	{	
		TCMOOutputFlowControlParms fcOpt;

		if (fHWType == k180si)
			fcOpt.useSoftFlowControl = true;
		else
		{
			fcOpt.useHardFlowControl = true;
			fcOpt.useSoftFlowControl = true;
		}

		result = options.AppendOption(&fcOpt);
	}

	if (result == noErr)				//	Create the endpoint
		result = CMGetEndpoint(&options, &fAsyncIntf, false);

	if (fAsyncIntf == nil || result != noErr)
		initFailed = true;
	else
		initFailed = (fAsyncIntf->EasyOpen(0) != noErr);

	if (initFailed)	
	{
		if (fAsyncIntf)
		{
			fAsyncIntf->Delete();
			fAsyncIntf = nil;
		}
	}

	PRINT(("End KodakInitConnection"));

	return !initFailed;
}


/*-----------------------------------------------------------------------------
	NewtonErr FaxEndPage(long pageCount)
	
	This method is used only by the fax driver.  For all others, leave empty.
	-----------------------------------------------------------------------------*/
NewtonErr DRIVERCLASSNAME::FaxEndPage(long /*pageCount*/)
{
	return noErr;
}